﻿using log4net;
using Microsoft.WindowsAzure.Storage;
using Microsoft.WindowsAzure.Storage.Blob;
using System;
using System.IO;
using System.Threading.Tasks;
using System.Xml.Serialization;
using VA.PPMS.IWS.BlobService.Interface;
using VA.PPMS.IWS.Common;
using VA.PPMS.IWS.Functions.Configuration.Interface;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.BlobService
{
    public class BlobService : IBlobService
    {
        private readonly ILog _logger;
        private readonly IIwsConfiguration _configuration;

        public BlobService(ILog logger, IIwsConfiguration configuration)
        {
            _logger = logger;
            _configuration = configuration;
        }

        public async Task<bool> PutBlobAsync(string transactionId, Providers providers)
        {
            try
            {
                _logger.Info($"@@@@ INFO - Start PutBlobAsync @@@@");
                var result = await SaveAsync(providers, transactionId);
                _logger.Info($"@@@@ INFO - End PutBlobAsync @@@@");

                return result;
            }
            catch (Exception ex)
            {
                _logger.Error($"!!!! ERROR - There was a problem with putting the Blob !!!!", ex);
                throw;
            }
        }

        public async Task<Providers> GetBlobAsync(string id)
        {
            try
            {
                _logger.Info($"@@@@ INFO - Start GetBlobAsync @@@@");
                var result = await GetAsync<Providers>(id);
                _logger.Info($"@@@@ INFO - End GetBlobAsync @@@@");

                return result;
            }
            catch (Exception ex)
            {
                _logger.Error($"!!!! ERROR - There was a problem with getting the Blob !!!!", ex);
                throw;
            }
        }

        private async Task<bool> SaveAsync<T>(T payload, string id)
        {
            var blobName = await _configuration.GetBlobNameAsync();
            var storageAccountString = await _configuration.GetConnectionAsync();

            var storageAccount = CloudStorageAccount.Parse(storageAccountString);
            var cloudBlobClient = storageAccount.CreateCloudBlobClient();
            var blobContainer = cloudBlobClient.GetContainerReference(blobName);
            blobContainer.SetPermissions(new BlobContainerPermissions { PublicAccess = BlobContainerPublicAccessType.Blob });
            blobContainer.CreateIfNotExists();

            var blockBlob = blobContainer.GetBlockBlobReference(id);
            blockBlob.Metadata["Status"] = "Processing";

            using (var stream = Serialize(payload))
            {
                await blockBlob.UploadFromStreamAsync(stream);
            }
                
            return true;
        }

        public async Task<T> GetAsync<T>(string id)
        {
            var blobName = await _configuration.GetBlobNameAsync();
            var storageAccountString = await _configuration.GetConnectionAsync();

            var storageAccount = CloudStorageAccount.Parse(storageAccountString);
            var cloudBlobClient = storageAccount.CreateCloudBlobClient();
            var blobContainer = cloudBlobClient.GetContainerReference(blobName);
            blobContainer.SetPermissions(new BlobContainerPermissions { PublicAccess = BlobContainerPublicAccessType.Blob });

            var blockBlob = blobContainer.GetBlockBlobReference(id);
            var xml = await blockBlob.DownloadTextAsync();

            return Deserialize<T>(xml);
        }

        private static Stream Serialize<T>(T person)
        {
            var serializer = new XmlSerializer(typeof(T));
            var stream = new MemoryStream();
            serializer.Serialize(stream, person);
            stream.Position = 0;

            return stream;
        }

        private static T Deserialize<T>(string xml)
        {
            var serializer = new XmlSerializer(typeof(T));

            using (TextReader reader = new StringReader(xml))
            {
                return (T)serializer.Deserialize(reader);
            }
        }
    }
}